//
//  ToyDetailViewController.swift
//  SwiftDemo
//
//  Created by Lovense on 2024/7/2.
//

import UIKit
import LovenseKit

let ScreenWidth = UIScreen.main.bounds.size.width
let ScreenHeight = UIScreen.main.bounds.size.height

class ToyDetailViewController: UIViewController, UITableViewDataSource, UITableViewDelegate {
    var currentToy: LovenseToy
    
    lazy var mainTableView: UITableView = {
        let _mainTableView = UITableView(frame: self.view.bounds, style: .plain)
        _mainTableView.backgroundColor = .white
        _mainTableView.scrollsToTop = true
        _mainTableView.delegate = self
        _mainTableView.dataSource = self
        _mainTableView.tableFooterView = .init()
        _mainTableView.tableHeaderView = self.tableHeaderView
        return _mainTableView
    }()
    lazy var tipsLabel: UILabel = {
        let _tipsLabel = UILabel()
        _tipsLabel.isHidden = true
        _tipsLabel.frame = .init(x: 20, y: StatusBarHeight + 200, width: (self.view.bounds.size.width - 40) , height: 30)
        return _tipsLabel
    }()
    let activityIndicator = UIActivityIndicatorView()
    lazy var tableHeaderView = ToyDetailHeaderView(frame: .init(x: 0, y: 0, width: self.view.bounds.size.width, height: 394))
    /// light indicator (e.g. Lush/Hush/Edge)/
    var isLightOn = false
    /// AID light indicator (e.g. Domi)/
    var isAidLightOn = false
    /// get battery timer
    var getBatteryTimer: Timer?
    
    /// Solace Trust Current value
    var solaceTrust = 0
    /// Solace Depth Current value
    var solaceDepth = 0
    
    /// MIssion2 Touch Mode Value
    var mission2TouchMode: Int = 0
    /// MIssion2 Touch Value
    var mission2TouchValue: [Int] = [0]
    /// MIssion2 Touch Level Value
    var mission2TouchLevel: Int = 0
    
    /// Solace Pro LVS Value
    var solaceProLVS: (speed: Int, start: Int, end: Int) = (0, 0, 0)
    /// Solace Pro Site Value
    var solaceProSite: (dir: Int, site: Int) = (0, 0)
    
    /// auto switch value
    var autoSwitch: (autoStop: Bool, reconnectToLastLevel: Bool) = (false, false)
    
    var lushLightColor: LushLightColorType?
    
    init(currentToy: LovenseToy) {
        self.currentToy = currentToy
        super.init(nibName: nil, bundle: nil)
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    override func viewWillDisappear(_ animated: Bool) {
        super.viewWillDisappear(animated)
        NotificationCenter.default.removeObserver(self)
        
        self.getBatteryTimer?.invalidate()
        self.getBatteryTimer = nil
        
        IQKeyboardManager.shared.enable = false
    }
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        IQKeyboardManager.shared.enable = true
    }

    override func viewDidLoad() {
        super.viewDidLoad()
        // get battery timer
        let timer = Timer(timeInterval: 5, target: self, selector: #selector(getBatteryHandler), userInfo: nil, repeats: true)
        RunLoop.current.add(timer, forMode: .common)
        self.getBatteryTimer = timer
        
        self.title = "\(self.currentToy.name)Toy Control"
        self.addNotification()
        self.initAll()
    }
    func addNotification() {
        NotificationCenter.default.addObserver(self, selector: #selector(connectSuccessCallback(_:)), name: .toyConnectSuccess, object: nil) // Toy connected
          
        NotificationCenter.default.addObserver(self, selector: #selector(connectFailCallback(_:)), name: .toyConnectFail, object: nil) // Failed to connect the toy
        
        NotificationCenter.default.addObserver(self, selector: #selector(connectBreakCallback(_:)), name: .toyConnectBreak, object: nil) // Toy is disconnected
        
        NotificationCenter.default.addObserver(self, selector: #selector(commandSuccessCallback(_:)), name: .toyCommandCallbackAtSuccess, object: nil) // Command sent successfully
        
        NotificationCenter.default.addObserver(self, selector: #selector(commandErrorCallback(_:)), name: .toyCommandCallbackAtError, object: nil) // Failed to send command
        
        NotificationCenter.default.addObserver(self, selector: #selector(deviceTypeCallback(_:)), name: .toyCallbackDeviceType, object: nil) // device information
        
        NotificationCenter.default.addObserver(self, selector: #selector(lightStatusCallback(_:)), name: .toyCallbackGetLightStatus, object: nil) // light indication
        
        NotificationCenter.default.addObserver(self, selector: #selector(aidLightStatusCallback(_:)), name: .toyCallbackGetAidLightStatus, object: nil) // AID light indication
        
        NotificationCenter.default.addObserver(self, selector: #selector(listenMove(_:)), name: .toyCallbackListenMove, object: nil) // Start tracking the toy movement (usually used in 2 way interactions)
        
        NotificationCenter.default.addObserver(self, selector: #selector(touchModeCallback(_:)), name: .toyCallbackTouchMode, object: nil) // touch mode
        NotificationCenter.default.addObserver(self, selector: #selector(touchValueCallback(_:)), name: .toyCallbackTouchValue, object: nil) // touch value
        NotificationCenter.default.addObserver(self, selector: #selector(touchLevelCallback(_:)), name: .toyCallbackTouchLevel, object: nil) // touch level
        NotificationCenter.default.addObserver(self, selector: #selector(touchInteractive(_:)), name: .toyCallbackTouchInteractive, object: nil) // collecting data
        NotificationCenter.default.addObserver(self, selector: #selector(getSiteCallback(_:)), name: .toyCallbackGetSite, object: nil) // get site
        NotificationCenter.default.addObserver(self, selector: #selector(getAutoSwitchCallback(_:)), name: .toyCallbackGetAutoSwitch, object: nil) // get auto switch
        
        NotificationCenter.default.addObserver(self, selector: #selector(toySendCommandError(_:)), name: .toySendCommandError, object: nil) // send command error
        
        NotificationCenter.default.addObserver(self, selector: #selector(toyGetCommandError(_:)), name: .toyGetCommandsError, object: nil) // send command error
        
        NotificationCenter.default.addObserver(self, selector: #selector(getToyBatteryCallback(_:)), name: .toyCallbackBattery, object: nil) // get battery
        
        NotificationCenter.default.addObserver(self, selector: #selector(getToyLightColorCallback(_:)), name: .toyCallbackGetLightColor, object: nil) // get light color
    }
    @objc func getBatteryHandler() {
        self.tableHeaderView.batteryLabel.text = "Battery:\(self.currentToy.battery)%"
    }
    func initAll() {
        self.view.backgroundColor = .white
        
        self.view.addSubview(self.mainTableView)
        self.tableHeaderView.showSupportedCommands = { [weak self] in
            guard let self else { return }
            if let commands = Lovense.shared.getSupportedCommands(self.currentToy.identifier) {
                let message = commands.map { $0.description }.joined(separator: "\n")
                let alertVc = UIAlertController(title: "Get Supported Commands", message: message, preferredStyle: .alert)
                let cancelBtn = UIAlertAction(title: "ok", style: .cancel)
                alertVc.addAction(cancelBtn)
                self.present(alertVc, animated: true)
            }
        }

        self.solaceDepth = 1
        self.solaceTrust = 0
        
        self.activityIndicator.style = .large
        self.activityIndicator.frame = .init(x: 0, y: 0, width: ScreenWidth, height: ScreenHeight)
        self.activityIndicator.backgroundColor = .black
        self.activityIndicator.alpha = 0.2
        self.activityIndicator.hidesWhenStopped = true
        self.view.addSubview(self.activityIndicator)
        
        self.loadRightButton()
        
        if self.currentToy.isConnected {
            self.tableHeaderView.batteryLabel.text = "Battery:\(self.currentToy.battery)%"
            self.tableHeaderView.macLabel.text = "MAC Address:\(self.currentToy.macAddress)"
            self.tableHeaderView.typeLabel.text = "Device Info:\(self.currentToy.toyType)"
            self.tableHeaderView.versionLabel.text = "Version:\(self.currentToy.version)"
            if let toyName = currentToy.toyName {
                self.tableHeaderView.toyNameLabel.text = "Toy Name:\(toyName)"
            }
            if let fullName = currentToy.fullName {
                self.tableHeaderView.fullNameLabel.text = "Full Name:\(fullName)"
            }
            if let toyVersion = currentToy.toyVersion {
                self.tableHeaderView.toyVersionLabel.text = "Toy Version:\(toyVersion)"
            }
        } else {
            //Connect a toy
            //step 4
            Lovense.shared.connectToy(self.currentToy.identifier)
            
            self.activityIndicator.startAnimating()
        }
        self.view.addSubview(self.tipsLabel)
    }
    @objc func onBreakClick() {
        Lovense.shared.disconnectToy(self.currentToy.identifier)
    }
    @objc func onConnectClick() {
        self.activityIndicator.startAnimating()
        Lovense.shared.connectToy(self.currentToy.identifier)
    }
    func loadRightButton() {
        if self.currentToy.isConnected {
            let rightBarItem = UIBarButtonItem(title: "Disconnect", style: .plain, target: self, action: #selector(onBreakClick))
            self.navigationItem.rightBarButtonItem = rightBarItem
        } else {
            let rightBarItem = UIBarButtonItem(title: "Connect", style: .plain, target: self, action: #selector(onConnectClick))
            self.navigationItem.rightBarButtonItem = rightBarItem
        }
    }
    func numberOfSections(in tableView: UITableView) -> Int {
        return 15
    }
    func tableView(_ tableView: UITableView, heightForHeaderInSection section: Int) -> CGFloat {
        if section == 8 {
            return 86
        }
        return 55
    }
    func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        switch section {
        case 0: return 11   //all
        case 1: return 4   //Nora
        case 2: return 2   //Edge/Gemini
        case 3: return 3   //Domi
        case 4: return 2   // Nora Max
        case 5: return 3   // Max
        case 6: return 2   // Flexer
        case 7: return 1   // Gravity/XMachine
        case 8: return 2   // Solace
        case 9: return 4   // Lapis
        case 10: return 8 // Mission2
        case 11: return 7 // Solace pro
        case 12: return 2 // Lush4
        case 13: return 1 // Gush2
        case 14: return 3 // Osci3
        default: return 0
        }
    }
    func tableView(_ tableView: UITableView, heightForRowAt indexPath: IndexPath) -> CGFloat {
        switch (indexPath.section, indexPath.row) {
        case  (6, 4), (9, 3): return 120 //Ambi, Domi, Osci
        default: return 90
        }
    }
    func tableView(_ tableView: UITableView, viewForHeaderInSection section: Int) -> UIView? {
        let headerView = UIView()
        headerView.backgroundColor = .systemGroupedBackground
        let label = UILabel()
        label.textColor = .black
        label.font = .systemFont(ofSize: 18)
        label.textAlignment = .center
        label.frame = .init(x: 15, y: 15, width: self.view.bounds.size.width, height: 20)
        headerView.addSubview(label)
        
        let subLabel = UILabel()
        subLabel.textColor = .black
        subLabel.font = .systemFont(ofSize: 16)
        subLabel.numberOfLines = 0
        subLabel.textAlignment = .center
        subLabel.frame = .init(x: 15, y: 40, width: self.view.bounds.size.width, height: 40)
        subLabel.isHidden = true
        headerView.addSubview(subLabel)
        
        switch section {
        case 0: label.text = "All Toy Command"
        case 1: label.text = "Nora/Ridge"
        case 2: label.text = "Edge/Gemini"
        case 3: label.text = "Domi"
        case 4: label.text = "Nora Max"
        case 5: label.text = "Max"
        case 6: label.text = "Flexer"
        case 7: label.text = "Gravity/XMachine/Mini XMachine"
        case 8:
            label.text = "Solace"
            subLabel.isHidden = false
            subLabel.text = "The Solace command requires both the Thrust and Depth settings"
        case 9: label.text = "Lapis"
        case 10: label.text = "Mission2"
        case 11: label.text = "Solace Pro"
        case 12: label.text = "Lush4"
        case 13: label.text = "Gush2"
        case 14: label.text = "Osci3"
        default: break
        }
        return headerView
    }
    func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {
        let identifier = "ToyDetailCell\(indexPath.section)_\(indexPath.row)"
        let cell: ToyDetailCell
        if let detailCell = tableView.dequeueReusableCell(withIdentifier: identifier) as? ToyDetailCell {
            cell = detailCell
        } else {
            tableView.register(ToyDetailCell.self, forCellReuseIdentifier: identifier)
            let detailCell = tableView.dequeueReusableCell(withIdentifier: identifier) as! ToyDetailCell
            detailCell.selectionStyle = .none
            cell = detailCell
        }
        
        switch (indexPath.section, indexPath.row) {
        case (0, 0):
            cell.setSliderName("Vibration")
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_VibrateLevel: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .vibrate, parameters: paramDict)
            }
        case (0, 1):
            cell.setBtnName("Quick Flash")
            cell.btnClickBlock = { [weak self] in
                guard let self else { return }
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .flash, parameters: nil)
            }
        case (0, 2):
            cell.setBtnName("Turn off Light")
            cell.btnClickBlock = { [weak self] in
                guard let self else { return }
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .lightOff, parameters: nil)
            }
        case (0, 3):
            cell.setBtnName("Turn on Light")
            cell.btnClickBlock = { [weak self] in
                guard let self else { return }
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .lightOn, parameters: nil)
            }
        case (0, 4):
            cell.setBtnName("Get Light Status=\(self.isLightOn)")
            cell.btnClickBlock = { [weak self] in
                guard let self else { return }
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .getLightStatus, parameters: nil)
            }
        case (0, 5):
            cell.setSliderName("Preset", min: 0, max: 10)
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_PresetLevel: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .preset, parameters: paramDict)
            }
        case (0, 6):
            cell.setBtnName("Turn on Auto Stop")
            cell.btnClickBlock = { [weak self] in
                guard let self else { return }
                self.autoSwitch.autoStop = true
                let paramDict = [kSendCommandParamKey_AutoStop: self.autoSwitch.autoStop,
                     kSendCommandParamKey_ReconnectToLastLevel: self.autoSwitch.reconnectToLastLevel]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .setAutoSwitch, parameters: paramDict)
            }
        case (0, 7):
            cell.setBtnName("Turn off Auto Stop")
            cell.btnClickBlock = { [weak self] in
                guard let self else { return }
                self.autoSwitch.autoStop = false
                let paramDict = [kSendCommandParamKey_AutoStop: self.autoSwitch.autoStop,
                     kSendCommandParamKey_ReconnectToLastLevel: self.autoSwitch.reconnectToLastLevel]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .setAutoSwitch, parameters: paramDict)
            }
        case (0, 8):
            cell.setBtnName("Turn on Reconnect To Last Level")
            cell.btnClickBlock = { [weak self] in
                guard let self else { return }
                self.autoSwitch.reconnectToLastLevel = true
                let paramDict = [kSendCommandParamKey_AutoStop: self.autoSwitch.autoStop,
                     kSendCommandParamKey_ReconnectToLastLevel: self.autoSwitch.reconnectToLastLevel]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .setAutoSwitch, parameters: paramDict)
            }
        case (0, 9):
            cell.setBtnName("Turn off Reconnect To Last Level")
            cell.btnClickBlock = { [weak self] in
                guard let self else { return }
                self.autoSwitch.reconnectToLastLevel = false
                let paramDict = [kSendCommandParamKey_AutoStop: self.autoSwitch.autoStop,
                     kSendCommandParamKey_ReconnectToLastLevel: self.autoSwitch.reconnectToLastLevel]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .setAutoSwitch, parameters: paramDict)
            }
        case (0, 10):
            cell.setBtnName("Get Auto Stop=\(self.autoSwitch.autoStop); \nReconnect To Last Level=\(self.autoSwitch.reconnectToLastLevel)")
            cell.btnClickBlock = { [weak self] in
                guard let self else { return }
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .getAutoSwitch, parameters: nil)
            }
        case (1, 0):
            cell.setSliderName("Nora Rotation")
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_RotateLevel: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .rotate, parameters: paramDict)
            }
        case (1, 1):
            cell.setSliderName("Nora Rotation")
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_RotateLevel: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .rotateClockwise, parameters: paramDict)
            }
        case (1, 2):
            cell.setSliderName("Nora Rotation(Anti-clockwise)")
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_RotateLevel: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .rotateAntiClockwise, parameters: paramDict)
            }
        case (1, 3):
            cell.setBtnName("Nora Rotation Change")
            cell.btnClickBlock = { [weak self] in
                guard let self else { return }
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .rotateChange, parameters: nil)
            }
        case (2, 0):
            cell.setSliderName("Edge/Gemini Vibrator 1")
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_VibrateLevel: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .vibrate1, parameters: paramDict)
            }
        case (2, 1):
            cell.setSliderName("Edge/Gemini Vibrator 2")
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_VibrateLevel: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .vibrate2, parameters: paramDict)
            }
        case (3, 0):
            cell.setBtnName("Turn off AID Light")
            cell.btnClickBlock = { [weak self] in
                guard let self else { return }
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .alightOff, parameters: nil)
            }
        case (3, 1):
            cell.setBtnName("Turn on AID Light")
            cell.btnClickBlock = { [weak self] in
                guard let self else { return }
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .alightOn, parameters: nil)
            }
        case (3, 2): // domi
            cell.setBtnName("Get AID Light Status=\(self.isAidLightOn)")
            cell.btnClickBlock = { [weak self] in
                guard let self else { return }
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .getAlightStatus, parameters: nil)
            }
        case (4, 0): // Nora Max
            cell.setBtnName("Start tracking movement")
            cell.btnClickBlock = { [weak self] in
                guard let self else { return }
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .startMove, parameters: nil)
            }
        case (4, 1): // Nora Max
            cell.setBtnName("Stop tracking movement")
            cell.btnClickBlock = { [weak self] in
                guard let self else { return }
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .stopMove, parameters: nil)
            }
        case (5, 0): // Max
            cell.setSliderName("Max Air In", min: 1, max: 3)
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_AirLevel: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .airIn, parameters: paramDict)
            }
        case (5, 1): // Max
            cell.setSliderName("Max Air Out", min: 1, max: 3)
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_AirLevel: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .airOut, parameters: paramDict)
            }
        case (5, 2): // Max
            cell.setSliderName("Max Air Auto", min: 0, max: 3)
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_AirAutoLevel: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .airAuto, parameters: paramDict)
            }
        case (6, 0):
            cell.setSliderName("Flexer Vibrate", min: 0, max: 20)
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_VibrateLevel: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .flexerVibrate, parameters: paramDict)
            }
        case (6, 1):
            cell.setSliderName("Flexer Finger", min: 0, max: 20)
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_FingerLevel: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .flexerFinger, parameters: paramDict)
            }
        case (7, 0):
            cell.setSliderName("Gravity/XMachine/Mini XMachine Thrust")
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_ThrustingLevel: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .thrust, parameters: paramDict)
            }
        case (8, 0):
            cell.setSliderName("Solace Thrust", min: 0, max: 20)
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                self.solaceTrust = value
                let paramDict = [kSendCommandParamKey_ThrustingLevel: value, kSendCommandParamKey_DepthLevel: self.solaceDepth]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .solace, parameters: paramDict)
            }
        case (8, 1):
            cell.setSliderName("Solace Depth", min: 1, max: 3)
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                self.solaceDepth = value
                let paramDict = [kSendCommandParamKey_ThrustingLevel: self.solaceTrust, kSendCommandParamKey_DepthLevel: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .solace, parameters: paramDict)
            }
        case (9, 0):
            cell.setSliderName("Lapis Vibrate 1", min: 0, max: 20)
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_VibrateLevel: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .vibrate1, parameters: paramDict)
            }
        case (9, 1):
            cell.setSliderName("Lapis Vibrate 2", min: 0, max: 20)
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_VibrateLevel: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .vibrate2, parameters: paramDict)
            }
        case (9, 2):
            cell.setSliderName("Lapis Vibrate 3", min: 0, max: 20)
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_VibrateLevel: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .vibrate3, parameters: paramDict)
            }
        case (9, 3):
            cell.setMultiInput(title: "Lapis Vibrate", placeHolders: ["-1", "-1", "-1"], sender: "SEND")
            cell.multiCommandBlock = { [weak self] values in
                guard let self else { return }
                self.view.endEditing(true)
                let paramDict = [kSendCommandParamKey_MultiplyLevel: values]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .multiply, parameters: paramDict)
            }
        case (10, 0):
            cell.setSliderName("Mission2 Touch Mode", min: 0, max: 5)
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_TouchMode: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .setTouchMode, parameters: paramDict)
            }
        case (10, 1):
            cell.setBtnName("Mission2 Touch Mode=\(self.mission2TouchMode)")
            cell.btnClickBlock = { [weak self] in
                guard let self else { return }
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .getTouchMode, parameters: nil)
            }
        case (10, 2):
            cell.setSliderName("Mission2 Touch Value 1", min: 0, max: 100)
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_TouchValue: 1, kSendCommandParamKey_TouchValue1: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .setTouchValue, parameters: paramDict)
            }
        case (10, 3):
            cell.setSliderName("Mission2 Touch Value 2", min: 0, max: 100)
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_TouchValue: 2, kSendCommandParamKey_TouchValue1: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .setTouchValue, parameters: paramDict)
            }
        case (10, 4):
            cell.setSliderName("Mission2 Touch Value 3", min: 0, max: 100)
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_TouchValue: 3, kSendCommandParamKey_TouchValue1: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .setTouchValue, parameters: paramDict)
            }
        case (10, 5):
            cell.setBtnName("Mission2 Touch Value=\(self.mission2TouchValue.map { $0.description }.joined(separator: ","))")
            cell.btnClickBlock = { [weak self] in
                guard let self else { return }
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .getTouchValue, parameters: nil)
            }
        case (10, 6):
            cell.setSliderName("Mission2 Touch Level", min: 1, max: 3)
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_TouchLevel: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .setTouchLevel, parameters: paramDict)
            }
        case (10, 7):
            cell.setBtnName("Mission2 Touch Level=\(self.mission2TouchLevel)")
            cell.btnClickBlock = { [weak self] in
                guard let self else { return }
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .getTouchLevel, parameters: nil)
            }
        case (11, 0):
            cell.setSliderName("Solace Pro LVS speed", min: 0, max: 20)
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                self.solaceProLVS.speed = value
                let paramDict = [kSendCommandParamKey_Speed: solaceProLVS.speed,
                                 kSendCommandParamKey_Start: solaceProLVS.start,
                                   kSendCommandParamKey_End: solaceProLVS.end]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .lvs, parameters: paramDict)
            }
        case (11, 1):
            cell.setSliderName("Solace Pro LVS start", min: 0, max: 20)
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                self.solaceProLVS.start = value
                let paramDict = [kSendCommandParamKey_Speed: solaceProLVS.speed,
                                 kSendCommandParamKey_Start: solaceProLVS.start,
                                   kSendCommandParamKey_End: solaceProLVS.end]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .lvs, parameters: paramDict)
            }
        case (11, 2):
            cell.setSliderName("Solace Pro LVS end", min: 0, max: 20)
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                self.solaceProLVS.end = value
                let paramDict = [kSendCommandParamKey_Speed: solaceProLVS.speed,
                                 kSendCommandParamKey_Start: solaceProLVS.start,
                                   kSendCommandParamKey_End: solaceProLVS.end]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .lvs, parameters: paramDict)
            }
        case (11, 3):
            cell.setSliderName("Solace Pro set site", min: 0, max: 100)
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_Site: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .setSite, parameters: paramDict)
            }
        case (11, 4):
            cell.setBtnName("Solace Pro dir = \(solaceProSite.dir), site = \(solaceProSite.site)")
            cell.btnClickBlock = { [weak self] in
                guard let self else { return }
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .getSite, parameters: nil)
            }
        case (11, 5):
            cell.setBtnName("Solace Pro start feedback")
            cell.btnClickBlock = { [weak self] in
                guard let self else { return }
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .startFeedback, parameters: nil)
            }
        case (11, 6):
            cell.setBtnName("Solace Pro end feedback")
            cell.btnClickBlock = { [weak self] in
                guard let self else { return }
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .endFeedback, parameters: nil)
                DispatchQueue.main.asyncAfter(deadline: .now() + 1) {
                    self.tableHeaderView.feedbackTextView.text = nil
                }
            }
        case (12, 0):
            cell.setSliderName("Set Light Color", min: 1, max: 15) { value in
                let color: Any = LushLightColorType(rawValue: value) ?? "unknow"
                return "\(color)"
            }
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_Value: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .setLightColor, parameters: paramDict)
            }
        case (12, 1):
            let color: Any = self.lushLightColor ?? "unknow"
            cell.setBtnName("Get Light Color=\(color)")
            cell.btnClickBlock = { [weak self] in
                guard let self else { return }
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .getLightColor, parameters: nil)
            }
        case (13, 0):
            cell.setSliderName("Oscillate / Vibrator / Vibrator 1")
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_Value: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .oscillate, parameters: paramDict)
            }
        case (14, 0):
            cell.setSliderName("Vibrator / Vibrator 1")
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_VibrateLevel: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .vibrate, parameters: paramDict)
            }
        case (14, 1):
            cell.setSliderName("Oscillate / Vibrator 2")
            cell.sliderChangeBlock = { [weak self] value in
                guard let self else { return }
                let paramDict = [kSendCommandParamKey_Value: value]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .oscillate, parameters: paramDict)
            }
        case (14, 2):
            cell.setMultiInput(title: "Osci3 Multiply", placeHolders: ["Vibrator", "Oscillate"], sender: "SEND")
            cell.multiCommandBlock = { [weak self] values in
                guard let self else { return }
                self.view.endEditing(true)
                let paramDict = [kSendCommandParamKey_MultiplyLevel: values]
                Lovense.shared.sendCommand(self.currentToy.identifier, command: .multiply, parameters: paramDict)
            }
        default:
            break
        }
        
        return cell
    }
    func tableView(_ tableView: UITableView, didSelectRowAt indexPath: IndexPath) {
        self.view.endEditing(true)
    }
    
    // MARK: Callbacks
    
    /// Toy Connected
    @objc func connectSuccessCallback(_ notification: Notification) {
        self.activityIndicator.stopAnimating()
        // Once connected, retrive the battery status
        let connectedSuccessDict = notification.object as? [String: Any]
        let connectedToy = connectedSuccessDict?["toy"] as? LovenseToy
        if let connectedToy, connectedToy.identifier == self.currentToy.identifier {
            self.currentToy = connectedToy
            if self.currentToy.isConnected {
                self.tableHeaderView.batteryLabel.text = "Battery:\(self.currentToy.battery)%"
                self.tableHeaderView.macLabel.text = "MAC Address:\(self.currentToy.macAddress)"
                self.tableHeaderView.typeLabel.text = "Device Info:\(self.currentToy.toyType)"
                self.tableHeaderView.versionLabel.text = "Version:\(self.currentToy.version)"
                if let toyName = currentToy.toyName {
                    self.tableHeaderView.toyNameLabel.text = "Toy Name:\(toyName)"
                }
                if let fullName = currentToy.fullName {
                    self.tableHeaderView.fullNameLabel.text = "Full Name:\(fullName)"
                }
                if let toyVersion = currentToy.toyVersion {
                    self.tableHeaderView.toyVersionLabel.text = "Toy Version:\(toyVersion)"
                }
            }
            self.loadRightButton()
        }
        tipsLabel.text = ""
    }
    /// Failed to connect the toy
    @objc func connectFailCallback(_ notification: Notification) {
        self.activityIndicator.stopAnimating()
        let resonDict = notification.object as? [String: Any] ?? [:]
        if let error = resonDict["error"] as? NSError {
            //debugPrint("code: \(error.code), mssage: \(error.localizedDescription)")
        }
        
        let alertVc = UIAlertController(title: nil, message: "connect fail=\(resonDict)", preferredStyle: .alert)
        let cancelBtn = UIAlertAction(title: "ok", style: .cancel)
        alertVc.addAction(cancelBtn)
        self.present(alertVc, animated: true)
    }
    /// Toy is disconnected
    @objc func connectBreakCallback(_ notification: Notification) {
        let resonDict = notification.object as? [String: Any] ?? [:]
        if let error = resonDict["error"] as? Error {
            // 如果没有error，代表是正常断开链接
            //debugPrint("异常断开的原因：\(error.localizedDescription)")
        }
        if let toy = resonDict["toy"] as? LovenseToy,
           toy.identifier == self.currentToy.identifier {
            self.activityIndicator.stopAnimating()
            let alertVc = UIAlertController(title: nil, message: "Toy is disconnected=\(resonDict)", preferredStyle: .alert)
            let cancelBtn = UIAlertAction(title: "ok", style: .cancel)
            alertVc.addAction(cancelBtn)
            self.present(alertVc, animated: true)
            self.loadRightButton()
        }
    }
    /// Command sent success
    @objc func commandSuccessCallback(_ notification: Notification) {
        let callbackDict = notification.object as? [String: Any]
        tipsLabel.text = ""
        //debugPrint("callbackDict==\(String(describing: callbackDict))")
    }
    /// Failed to send command
    @objc func commandErrorCallback(_ notification: Notification) {
        let callbackDict = notification.object as? [String: Any]
        if let error = callbackDict?["error"] as? NSError {
            // 如果没有error，代表是正常断开链接
            //debugPrint("code: \(error.code), mssage: \(error.localizedDescription)")
        }
        if let resonDict = notification.object as? [String: Any],
             let error = resonDict["error"] as? NSError {
            // 如果没有error，代表是正常断开链接
            tipsLabel.text = "order exc - code: \(error.code), mssage: \(error.localizedDescription)"
        }
        //debugPrint("callbackDict==\(String(describing: callbackDict))")
    }
    /// Device information
    @objc func deviceTypeCallback(_ notification: Notification) {
        let callbackDict = notification.object as? [String: Any]
        //debugPrint("callbackDict==\(String(describing: callbackDict))")
        let receiveToy = callbackDict?["receiveToy"] as? LovenseToy
        if self.currentToy === receiveToy {
            self.tableHeaderView.macLabel.text = "MAC Address:\(self.currentToy.macAddress)"
            self.tableHeaderView.typeLabel.text = "Device Info:\(self.currentToy.toyType)"
            self.tableHeaderView.versionLabel.text = "Version:\(self.currentToy.version)"
        }
    }
    /// Light indicator
    @objc func lightStatusCallback(_ notification: Notification) {
        let callbackDict = notification.object as? [String: Any]
        self.isLightOn = callbackDict?["isOpen"] as? Bool ?? false
        self.mainTableView.reloadData()
    }
    /// AID Light Indicator
    @objc func aidLightStatusCallback(_ notification: Notification) {
        let callbackDict = notification.object as? [String: Any]
        self.isAidLightOn = callbackDict?["isOpen"] as? Bool ?? false
        self.mainTableView.reloadData()
    }
    /// Start tracking movement
    @objc func listenMove(_ notification: Notification) {
        let callbackDict = notification.object as? [String: Any]
        let level = callbackDict?["moveLevel"]
        self.tableHeaderView.moveLevelLabel.text = "Movement:\(String(describing: level))"
    }
    /// Touch mode
    @objc func touchModeCallback(_ notification: Notification) {
        let callbackDict = notification.object as? [String: Any]
        self.mission2TouchMode = callbackDict?["touchMode"] as? Int ?? 0
        self.mainTableView.reloadData()
    }
    /// Touch value
    @objc func touchValueCallback(_ notification: Notification) {
        let callbackDict = notification.object as? [String: Any]
        self.mission2TouchValue = callbackDict?["touchValue"] as? [Int] ?? [0]
        self.mainTableView.reloadData()
    }
    /// Touch level
    @objc func touchLevelCallback(_ notification: Notification) {
        let callbackDict = notification.object as? [String: Any]
        self.mission2TouchLevel = callbackDict?["touchLevel"] as? Int ?? 0
        self.mainTableView.reloadData()
    }
    /// Touch interactive data
    @objc func touchInteractive(_ notification: Notification) {
        if let callbackDict = notification.object as? [String: Any],
           let data = callbackDict["data"] as? [[String: Any]] {
            let datas = data.map { "dir: \($0["dir"] ?? 0), speed: \($0["speed"] ?? 0), site: \($0["site"] ?? 0)" }
            self.tableHeaderView.feedbackTextView.text = datas.joined(separator: "\n")
        }
    }
    /// get site
    @objc func getSiteCallback(_ notification: Notification) {
        let callbackDict = notification.object as? [String: Any]
        self.solaceProSite = (callbackDict?["dir"] as? Int ?? 0, callbackDict?["site"] as? Int ?? 0)
        self.mainTableView.reloadData()
    }
    /// get auto switch
    @objc func getAutoSwitchCallback(_ notification: Notification) {
        let callbackDict = notification.object as? [String: Any]
        self.autoSwitch = (callbackDict?["autoStop"] as? Bool ?? false, callbackDict?["reconnectToLastLevel"] as? Bool ?? false)
        self.mainTableView.reloadData()
    }
    
    /// get battery
    @objc func getToyBatteryCallback(_ notification: Notification) {
        let callbackDict = notification.object as? [String: Any]
        let battery = callbackDict?["battery"] as? Int ?? 0
        let isWork = callbackDict?["isWork"] as? Bool ?? false
        self.tableHeaderView.batteryLabel.text = "Battery:\(battery)% isWork:\(isWork ? "震动中" : "待机中")"
        self.mainTableView.reloadData()
    }
    
    /// get light color
    @objc func getToyLightColorCallback(_ notification: Notification) {
        let callbackDict = notification.object as? [String: Any]
        let lightColor = callbackDict?["lightColor"] as? Int ?? 0
        self.lushLightColor = LushLightColorType(rawValue: lightColor)
        self.mainTableView.reloadData()
    }
    
    /// send command error
    @objc func toySendCommandError(_ notification: Notification) {
        if let resonDict = notification.object as? [String: Any],
             let error = resonDict["error"] as? NSError {
            // 如果没有error，代表是正常断开链接
            tipsLabel.text = "sendOrder - code: \(error.code), mssage: \(error.localizedDescription)"
        }
    }
    /// get command error
    @objc func toyGetCommandError(_ notification: Notification) {
        if let resonDict = notification.object as? [String: Any],
             let error = resonDict["error"] as? NSError {
            // 如果没有error，代表是正常断开链接
            tipsLabel.text = "get supported commands - error: \(error.localizedDescription)"
        }
    }
}
extension LovenseCommandType: CustomStringConvertible {
    public var description: String {
        switch self {
        case .vibrate: return "vibrate"
        case .rotate: return "rotate"
        case .rotateClockwise: return "rotateClockwise"
        case .rotateAntiClockwise: return "rotateAntiClockwise"
        case .rotateChange: return "rotateChange"
        case .airIn: return "airIn"
        case .airOut: return "airOut"
        case .airAuto: return "airAuto"
        case .preset: return "preset"
        case .vibrate1: return "vibrate1"
        case .vibrate2: return "vibrate2"
        case .vibrate3: return "vibrate3"
        case .vibrateFlash: return "vibrateFlash"
        case .flash: return "flash"
        case .lightOff: return "lightOff"
        case .lightOn: return "lightOn"
        case .getLightStatus: return "getLightStatus"
        case .alightOff: return "alightOff"
        case .alightOn: return "alightOn"
        case .getAlightStatus: return "getAlightStatus"
        case .getBattery: return "getBattery"
        case .getDeviceType: return "getDeviceType"
        case .startMove: return "startMove"
        case .stopMove: return "stopMove"
        case .flexerVibrate: return "flexerVibrate"
        case .flexerFinger: return "flexerFinger"
        case .thrust: return "thrust"
        case .solace: return "solace"
        case .getTouchMode: return "getTouchMode"
        case .setTouchMode: return "setTouchMode"
        case .getTouchValue: return "getTouchValue"
        case .setTouchValue: return "setTouchValue"
        case .getTouchLevel: return "getTouchLevel"
        case .setTouchLevel: return "setTouchLevel"
        case .lvs: return "lvs"
        case .getSite: return "getSite"
        case .setSite: return "setSite"
        case .startFeedback: return "startFeedback"
        case .endFeedback: return "endFeedback"
        case .getAutoSwitch: return "getAutoSwitch"
        case .setAutoSwitch: return "setAutoSwitch"
        case .multiply: return "multiply"
        default: return self.rawValue.description
        }
    }
}
