//
//  SearchToyViewController.swift
//  SwiftDemo
//
//  Created by Lovense on 2024/7/2.
//

import UIKit
import CoreBluetooth
import LovenseKit

let StatusBarHeight = UIApplication.shared.statusBarFrame.size.height


extension SearchToyViewController: UITextFieldDelegate {
    func textFieldShouldReturn(_ textField: UITextField) -> Bool {
        sendButtonTapped()
        textField.resignFirstResponder()
        return true
    }
}

class SearchToyViewController: UIViewController, CBCentralManagerDelegate, UITableViewDataSource, UITableViewDelegate {

    lazy var textField: UITextField = {
        let _textField = UITextField()
        _textField.placeholder = "请粘贴输入Token内容"
        _textField.borderStyle = .roundedRect
        _textField.clearButtonMode = .whileEditing
        _textField.returnKeyType = .done
//        textField.translatesAutoresizingMaskIntoConstraints = false
        _textField.delegate = self
        let viewWidth = self.view.bounds.size.width
        let buttonWidth = (viewWidth - 60)
        let buttonX = viewWidth - 20 - buttonWidth
        _textField.frame = .init(x: 30, y: StatusBarHeight + 40, width: buttonWidth, height: 44)
        return _textField
    }()
    
    lazy var centralManager = CBCentralManager(delegate: self, queue: .main)
    lazy var mainTableView: UITableView = {
        let _mainTableView = UITableView(frame: .init(x: 0, y: StatusBarHeight + 230, width: self.view.bounds.size.width, height: self.view.bounds.size.height - 230))
        _mainTableView.backgroundColor = .white
        _mainTableView.scrollsToTop = true
        _mainTableView.delegate = self
        _mainTableView.dataSource = self
        _mainTableView.tableFooterView = .init()
        return _mainTableView
    }()
    lazy var loadBtn: UIButton = {
        let _searchBtn = UIButton(type: .system)
        _searchBtn.frame = .init(x: 20, y: StatusBarHeight + 120, width: (self.view.bounds.size.width - 60) * 0.5, height: 44)
        _searchBtn.setTitle("load toys", for: .normal)
        _searchBtn.layer.borderColor = UIColor.blue.cgColor
        _searchBtn.layer.cornerRadius = 4
        _searchBtn.layer.borderWidth = 1
        _searchBtn.layer.masksToBounds = true
        _searchBtn.addTarget(self, action: #selector(onLoadToys), for: .touchUpInside)
        return _searchBtn
    }()
    lazy var saveBtn: UIButton = {
        let viewWidth = self.view.bounds.size.width
        let buttonWidth = (viewWidth - 60) * 0.5
        let buttonX = viewWidth - 20 - buttonWidth
        
        let _searchBtn = UIButton(type: .system)
        _searchBtn.frame = .init(x: buttonX, y: StatusBarHeight + 120, width: buttonWidth, height: 44)
        _searchBtn.setTitle("save toys", for: .normal)
        _searchBtn.layer.borderColor = UIColor.blue.cgColor
        _searchBtn.layer.cornerRadius = 4
        _searchBtn.layer.borderWidth = 1
        _searchBtn.layer.masksToBounds = true
        _searchBtn.addTarget(self, action: #selector(onSaveToys), for: .touchUpInside)
        return _searchBtn
    }()
    lazy var searchBtn: UIButton = {
        let _searchBtn = UIButton(type: .system)
        _searchBtn.frame = .init(x: 20, y: StatusBarHeight + 170, width: (self.view.bounds.size.width - 60) * 0.5, height: 44)
        _searchBtn.setTitle("search", for: .normal)
        _searchBtn.layer.borderColor = UIColor.blue.cgColor
        _searchBtn.layer.cornerRadius = 4
        _searchBtn.layer.borderWidth = 1
        _searchBtn.layer.masksToBounds = true
        _searchBtn.addTarget(self, action: #selector(onScanClick), for: .touchUpInside)
        return _searchBtn
    }()
    lazy var stopSearchBtn: UIButton = {
        let viewWidth = self.view.bounds.size.width
        let buttonWidth = (viewWidth - 60) * 0.5
        let buttonX = viewWidth - 20 - buttonWidth
        
        let _stopSearchBtn = UIButton(type: .system)
        _stopSearchBtn.frame = .init(x: buttonX, y: StatusBarHeight + 170, width: buttonWidth, height: 44)
        _stopSearchBtn.setTitle("stop search", for: .normal)
        _stopSearchBtn.layer.borderColor = UIColor.blue.cgColor
        _stopSearchBtn.layer.cornerRadius = 4
        _stopSearchBtn.layer.borderWidth = 1
        _stopSearchBtn.layer.masksToBounds = true
        _stopSearchBtn.addTarget(self, action: #selector(onStopScanClick), for: .touchUpInside)
        return _stopSearchBtn
    }()
    lazy var bluetoothStatusLabel: UILabel = {
        let _bluetoothStatusLabel = UILabel()
        _bluetoothStatusLabel.frame = .init(x: 20, y: StatusBarHeight + 80, width: (self.view.bounds.size.width - 40) * 0.5, height: 30)
        return _bluetoothStatusLabel
    }()
    lazy var tipsLabel: UILabel = {
        let _tipsLabel = UILabel()
        _tipsLabel.frame = .init(x: self.view.bounds.size.width * 0.5 + 20, y: StatusBarHeight + 80, width: (self.view.bounds.size.width - 40) * 0.55, height: 30)
        return _tipsLabel
    }()
    var allToyModelArr: [LovenseToy] = []
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        NotificationCenter.default.removeObserver(self)
        // step 3 (scanSuccess)
        NotificationCenter.default.addObserver(self, selector: #selector(scanSuccessCallback(_:)), name: .toyScanSuccess, object: nil) // Scanning toy success notification
        NotificationCenter.default.addObserver(self, selector: #selector(connectSuccessCallback(_:)), name: .toyConnectSuccess, object: nil) // Connected toy successfully notification
        NotificationCenter.default.addObserver(self, selector: #selector(connectFailCallback(_:)), name: .toyConnectFail, object: nil) // Failed to connect the toy
        NotificationCenter.default.addObserver(self, selector: #selector(connectBreakCallback(_:)), name: .toyConnectBreak, object: nil) // Toy is disconnected
        NotificationCenter.default.addObserver(self, selector: #selector(toySendCommandError(_:)), name: .toySendCommandError, object: nil) // send command error
        NotificationCenter.default.addObserver(self, selector: #selector(commandSuccessCallback(_:)), name: .toyCommandCallbackAtSuccess, object: nil) // Command sent successfully

        NotificationCenter.default.addObserver(self, selector: #selector(commandErrorCallback(_:)), name: .toyCommandCallbackAtError, object: nil) // Failed to send command
        
        self.mainTableView.reloadData()
    }
    override func viewDidLoad() {
        super.viewDidLoad()
        self.title = "Lovense Toy"
        
        // step 1 (setDeveloperToken)
        let token: String = ""
        verifyToken(with: token)
        
#if DEBUG
        // 开启日志打印[默认： 关闭]
        Lovense.shared.setLogEnable(enable: true)
#endif
        _ = self.centralManager
        
        self.initView()
    }
    
    private func verifyToken(with token: String?) {
        guard let token = token,
              token.isEmpty == false else {
            //debugPrint("\(#function) [Line \(#line)] please input your token!")
            return
        }
        
        Lovense.shared.setDeveloperToken(token)
    }
    
    @objc private func sendButtonTapped() {
        verifyToken(with: textField.text)
//          if let text = textField.text, !text.isEmpty {
//              //debugPrint("发送内容: \(text)")
//              textField.text = ""
//          }
      }
    
    func initView() {
        self.navigationController?.interactivePopGestureRecognizer?.isEnabled = false
        self.view.backgroundColor = .systemGroupedBackground
        self.view.addSubview(self.mainTableView)
        self.view.addSubview(self.searchBtn)
        self.view.addSubview(self.stopSearchBtn)
        self.view.addSubview(self.loadBtn)
        self.view.addSubview(self.saveBtn)
        self.view.addSubview(self.bluetoothStatusLabel)
        self.view.addSubview(self.tipsLabel)
        self.view.addSubview(self.textField)
        
        let rightBarItem = UIBarButtonItem(title: "clear", style: .plain, target: self, action: #selector(clearAllToy))
        self.navigationItem.rightBarButtonItem = rightBarItem
    }
    
    @objc func onSaveToys() {
        // Save a toy to the local list
        Lovense.shared.saveToys(self.allToyModelArr)
        self.tipsLabel.text = "save toys"
    }
    @objc func onLoadToys() {
        self.allToyModelArr = Lovense.shared.listToys()
        // Retrieve the saved toys
        self.mainTableView.reloadData()
        self.tipsLabel.text = "load toys"
    }
    func centralManagerDidUpdateState(_ central: CBCentralManager) {
        switch central.state {
        case .poweredOn:
            self.bluetoothStatusLabel.text = "bluetooth: 🔵"
        default:
            self.bluetoothStatusLabel.text = "bluetooth: ⚪️"
            for (i, toy) in self.allToyModelArr.enumerated() {
                let cell = self.mainTableView.cellForRow(at: .init(row: i, section: 0))
                let notConectStr = "not connected ❌"
                
                cell?.textLabel?.text = "name:\(toy.name)\nID:\(toy.identifier)\nRSSI:\(toy.rssi)\nstatus:\(notConectStr)"
            }
        }
    }
    
    @objc func onScanClick() {
        let alert = UIAlertController()
        let autoConnect = UIAlertAction(title: "Auto Connect", style: .default) { [weak self] _ in
            guard let self else { return }
            // step 2
            Lovense.shared.searchToys(autoConnect: true)
            self.tipsLabel.text = "searching toy"
        }
        let nonAutoConnect = UIAlertAction(title: "Do not Auto Connect", style: .default) { [weak self] _ in
            guard let self else { return }
            // step 2
            Lovense.shared.searchToys(autoConnect: false)
            self.tipsLabel.text = "searching toy"
        }
        alert.addAction(autoConnect)
        alert.addAction(nonAutoConnect)
        self.present(alert, animated: true)
    }
    @objc func onStopScanClick() {
        Lovense.shared.stopSearching()
        self.tipsLabel.text = "stop search toy"
    }
    
    @objc func clearAllToy() {
        self.allToyModelArr.removeAll()
        self.mainTableView.reloadData()
        
        self.tipsLabel.text = "clear all toy"
    }
    
    func numberOfSections(in tableView: UITableView) -> Int {
        return 1
    }
    func tableView(_ tableView: UITableView, heightForHeaderInSection section: Int) -> CGFloat {
        return 0
    }
    func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        return self.allToyModelArr.count
    }
    func tableView(_ tableView: UITableView, heightForRowAt indexPath: IndexPath) -> CGFloat {
        return 90
    }
    func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {
        let identifier = "SearchToyCell\(indexPath.section)_\(indexPath.row)"
        let cell: UITableViewCell
        if let detailCell = tableView.dequeueReusableCell(withIdentifier: identifier) {
            cell = detailCell
        } else {
            tableView.register(ToyDetailCell.self, forCellReuseIdentifier: identifier)
            let detailCell = tableView.dequeueReusableCell(withIdentifier: identifier)!
            detailCell.selectionStyle = .none
            cell = detailCell
        }
        
        let toy = self.allToyModelArr[indexPath.row]
        cell.textLabel?.numberOfLines = 0
        var isConectStr = "not connected ❌"
        if toy.isConnected {
            isConectStr = "connected ✅"
        }
        
        cell.textLabel?.text = "name:\(toy.name)\nID:\(toy.identifier)\nRSSI:\(toy.rssi)\nstatus:\(isConectStr)"
        cell.textLabel?.font = .systemFont(ofSize: 14)
        
        return cell
    }
    func tableView(_ tableView: UITableView, didSelectRowAt indexPath: IndexPath) {
        let toy = self.allToyModelArr[indexPath.row]
        
        let detailVC = ToyDetailViewController(currentToy: toy)
        self.navigationController?.pushViewController(detailVC, animated: true)
    }
    func tableView(_ tableView: UITableView, commit editingStyle: UITableViewCell.EditingStyle, forRowAt indexPath: IndexPath) {
        if editingStyle == .delete {
            let toy = self.allToyModelArr[indexPath.row]
            Lovense.shared.removeToy(toy.identifier)
            self.allToyModelArr.remove(at: indexPath.row)
            self.mainTableView.reloadData()
        }
    }
        
    // MARK: Callback
    
    /// step 3
    @objc func scanSuccessCallback(_ notification: Notification) {
        if  let dict = notification.object as? [String: Any],
            let scanToyArr = dict["scanToyArray"] as? [LovenseToy] {
            self.allToyModelArr = scanToyArr
            self.mainTableView.reloadData()
        }
        tipsLabel.text = ""
    }
    /// connectSuccess
    @objc func connectSuccessCallback(_ notification: Notification) {
        if let dict = notification.object as? [String: Any],
           let connectedToy = dict["toy"] as? LovenseToy,
           let toyIndex = self.allToyModelArr.firstIndex(where: { $0.identifier == connectedToy.identifier }) {
            self.allToyModelArr.remove(at: toyIndex)
            self.allToyModelArr.append(connectedToy)
            
            //debugPrint("\(connectedToy)")
            self.mainTableView.reloadData()
        }
        tipsLabel.text = ""
    }
    /// Failed to connect the toy
    @objc func connectFailCallback(_ notification: Notification) {
        let resonDict = notification.object as? [String: Any] ?? [:]
        if let error = resonDict["error"] as? NSError {
            tipsLabel.text = "connectFail - code: \(error.code), mssage: \(error.localizedDescription)"
        }
        let alertVc = UIAlertController(title: nil, message: "connect fail=\(resonDict)", preferredStyle: .alert)
        let cancelBtn = UIAlertAction(title: "ok", style: .cancel)
        alertVc.addAction(cancelBtn)
        self.present(alertVc, animated: true)
        
    }
    /// Toy is disconnected
    @objc func connectBreakCallback(_ notification: Notification) {
        if  let resonDict = notification.object as? [String: Any],
            let breakToy = resonDict["toy"] as? LovenseToy,
            let toyIndex = self.allToyModelArr.firstIndex(where: { $0.identifier == breakToy.identifier}) {
            if let error = resonDict["error"] as? NSError {
                // 如果没有error，代表是正常断开链接
                tipsLabel.text = "disconnect code: \(error.code), mssage: \(error.localizedDescription)"
            }
            let toy = self.allToyModelArr[toyIndex]
            let notConectStr = "not connected ❌"
            let cell = self.mainTableView.cellForRow(at: .init(row: toyIndex, section: 0))
            cell?.textLabel?.text = "name:\(toy.name)\nID:\(toy.identifier)\nRSSI:\(toy.rssi)\nstatus:\(notConectStr)"
        }
    }
    
    /// send command error
    @objc func toySendCommandError(_ notification: Notification) {
        if let resonDict = notification.object as? [String: Any],
             let error = resonDict["error"] as? NSError {
            // 如果没有error，代表是正常断开链接
            tipsLabel.text = "sendOrder - code: \(error.code), mssage: \(error.localizedDescription)"
        }
    }
    
    /// command exc error
    @objc func commandErrorCallback(_ notification: Notification) {
        if let resonDict = notification.object as? [String: Any],
             let error = resonDict["error"] as? NSError {
            // 如果没有error，代表是正常断开链接
            tipsLabel.text = "order exc - code: \(error.code), mssage: \(error.localizedDescription)"
        }
    }
    
    /// Command sent success
    @objc func commandSuccessCallback(_ notification: Notification) {
        let callbackDict = notification.object as? [String: Any]
        tipsLabel.text = ""
        //debugPrint("callbackDict==\(String(describing: callbackDict))")
    }
}
