//
//  ExampleForPatternViewController.swift
//  LovenseDemo
//
//  Created by 范晓鑫 on 2024/8/5.
//

import UIKit
import AVFoundation
import LovenseKit

class ExampleForPatternViewController: UIViewController, UITextFieldDelegate {
    /// player
    var player: AVPlayer? = nil
    /// playItem
    var playerItem: AVPlayerItem? = nil
    /// playerLayer
    var playerLayer: AVPlayerLayer? = nil
    /// current Time
    let currentTimeLabel: UILabel = UILabel()
    /// total Time
    let totalTimeLabel: UILabel = UILabel()
    /// Sliser
    let slider: UISlider = UISlider()
    /// Rate
    var rate: Float = 1
    var isDragSlider: Bool = false
    let textField: UITextField = UITextField()
    
    var _timeObserver: Any?
    
    deinit {
        self.player?.pause()
        self.player = nil
        if let _timeObserver {
            self.player?.removeTimeObserver(_timeObserver)
            self._timeObserver = nil
        }
    }
    override func viewDidLoad() {
        super.viewDidLoad()
        self.setupUI()
        
        self.rate = 1
        // Initialize your toys and platform information
        let toyIds: [String] = []
        
        Lovense.shared.setupPatternPlayer(toyIds: toyIds, pf: "Your platform")
        
        let interval = CMTimeMakeWithSeconds(1, preferredTimescale: Int32(NSEC_PER_SEC))
        self._timeObserver = self.player?.addPeriodicTimeObserver(forInterval: interval, queue: .main, using: { [weak self] time in
            guard let self else { return }
            if let loadedRanges = self.playerItem?.seekableTimeRanges,
               loadedRanges.count > 0 {
                if self.isDragSlider {
                    self.isDragSlider = false
                } else {
                    let currentTime = self.currentTime()
                    let totalTime = self.totalTime()
                    if currentTime == totalTime {
                        self.slider.value = 0
                        self.player?.seek(to: .zero)
                        self.player?.pause()
                        self.currentTimeLabel.text = "0:00"
                        Lovense.shared.stopPattern()
                    } else {
                        self.slider.value = Float(currentTime / totalTime)
                        self.currentTimeLabel.text = self.secondsConvertedToMinuteString(seconds: Int(currentTime))
                        self.totalTimeLabel.text = self.secondsConvertedToMinuteString(seconds: Int(totalTime))
                    }
                }
            }
        })
    }
    func setupUI() {
        self.view.backgroundColor = .white
        
        self.title = "Pattern Play"
        self.playerItem = AVPlayerItem(url: URL(string: "Your media URL")!)
        self.player = AVPlayer(playerItem: self.playerItem)
        let playerLayer = AVPlayerLayer(player: self.player)
        playerLayer.frame = CGRect(x: 0, y: 80, width: self.view.bounds.size.width, height: 200)
        playerLayer.backgroundColor = UIColor.gray.cgColor
        self.view.layer.addSublayer(playerLayer)
        self.playerLayer = playerLayer
        
        let playButton = UIButton()
        playButton.setTitle("Play", for: .normal)
        playButton.setTitleColor(.blue, for: .normal)
        playButton.setTitleColor(.red, for: .highlighted)
        playButton.layer.borderColor = UIColor.red.cgColor
        playButton.layer.cornerRadius = 3
        playButton.layer.borderWidth = 1
        playButton.titleLabel?.font = .systemFont(ofSize: 15)
        self.view.addSubview(playButton)
        playButton.addTarget(self, action: #selector(clickPlayButton), for: .touchUpInside)
        playButton.frame = CGRect(x: 50, y: 300, width: 80, height: 40)
        
        let pauseButton = UIButton()
        pauseButton.setTitle("Pause", for: .normal)
        pauseButton.setTitleColor(.blue, for: .normal)
        pauseButton.setTitleColor(.red, for: .highlighted)
        pauseButton.titleLabel?.font = .systemFont(ofSize: 15)
        pauseButton.layer.borderColor = UIColor.red.cgColor
        pauseButton.layer.cornerRadius = 3
        pauseButton.layer.borderWidth = 1
        self.view.addSubview(pauseButton)
        pauseButton.addTarget(self, action: #selector(clickPauseButton), for: .touchUpInside)
        pauseButton.frame = CGRect(x: 150, y: 300, width: 80, height: 40);
        
        let stopButton = UIButton()
        stopButton.setTitle("Stop", for: .normal)
        stopButton.setTitleColor(.blue, for: .normal)
        stopButton.setTitleColor(.red, for: .highlighted)
        stopButton.titleLabel?.font = .systemFont(ofSize: 15)
        stopButton.layer.borderColor = UIColor.red.cgColor
        stopButton.layer.cornerRadius = 3
        stopButton.layer.borderWidth = 1
        self.view.addSubview(stopButton)
        stopButton.addTarget(self, action: #selector(clickStopButton), for: .touchUpInside)
        stopButton.frame = CGRect(x: 250, y: 300, width: 80, height: 40);
        
        let label = UILabel()
        label.text = "Set Rate(0 ~ 2.0):"
        label.textColor = .black
        self.view.addSubview(label)
        label.frame = CGRect(x: 20, y: 370, width: 150, height: 40)
        
        let textField = self.textField
        textField.keyboardType = .numbersAndPunctuation
        textField.returnKeyType = .done
        textField.placeholder = "Default is 1.0"
        textField.delegate = self
        textField.borderStyle = .roundedRect
        textField.textColor = .blue
        self.view.addSubview(textField)
        textField.frame = CGRect(x: 20, y: 415, width: 220, height: 40)
        
        let label1 = UILabel()
        label1.text = "Drag here to adjust the playback"
        label1.textColor = .black
        self.view.addSubview(label1)
        label1.frame = CGRect(x: 20, y: 470, width: 300, height: 40)
        
        let currentTimeLabel = self.currentTimeLabel
        currentTimeLabel.textAlignment = .right
        currentTimeLabel.text = "--:--"
        currentTimeLabel.textColor = .black
        self.view.addSubview(currentTimeLabel)
        currentTimeLabel.frame = CGRect(x: 10, y: 517, width: 47, height: 20)
        
        let slider = self.slider
        slider.minimumValue = 0
        slider.maximumValue = 1
        slider.value = 0
        slider.addTarget(self, action: #selector(sliderValueChanged(_:)), for: .touchUpInside)
        self.view.addSubview(slider)
        slider.frame = CGRect(x: 70, y: 510, width: 230, height: 40)
        
        let totalTimeLabel = self.totalTimeLabel
        totalTimeLabel.textAlignment = .left
        totalTimeLabel.text = "--:--"
        totalTimeLabel.textColor = .black
        self.view.addSubview(totalTimeLabel)
        totalTimeLabel.frame = CGRect(x: 310, y: 517, width: 50, height: 20)
    }
    
    // MARK: Actions
    
    @objc func clickPlayButton() {
        self.player?.play()
        self.player?.rate = self.rate
        
        Lovense.shared.stopPattern()
        Lovense.shared.setRate(self.rate)
        /// prepare pattern file
        Lovense.shared.prepareForPattern(mediaId: "Your mediaId") { [weak self] state in
            guard let self else { return }
            switch state {
            case .downloading: break
            case .downloadCompleted:
                // play pattern
                Lovense.shared.playPattern(currentTime: self.currentTime() * 1000)
            case .downloadFail:
                // do something
                let alert = UIAlertController(title: "notice", message: "pattern file download fail", preferredStyle: .alert)
                let action1 = UIAlertAction(title: "OK", style: .cancel)
                alert.addAction(action1)
                self.present(alert, animated: true, completion: nil)
            @unknown default: break
            }
        }
    }
    @objc func clickPauseButton() {
        self.player?.pause()
        Lovense.shared.pausePattern()
    }
    @objc func clickStopButton() {
        self.player?.pause()
        self.player?.seek(to: .zero)
        Lovense.shared.stopPattern()
    }
    @objc func sliderValueChanged(_ slider: UISlider) {
        let progress = slider.value / slider.maximumValue;
        let currentTime = self.totalTime() * TimeInterval(progress)
        self.player?.seek(to: CMTimeMake(value: Int64(currentTime), timescale: 1))
        // sync current time
        if self.player?.timeControlStatus == .paused {
            self.clickPlayButton()
        } else {
            self.isDragSlider = true
            Lovense.shared.syncCurrentTime(currentTime * 1000)
        }
    }
    
    // MARK: UITextFieldDelegate
    
    func textFieldShouldReturn(_ textField: UITextField) -> Bool {
        textField.resignFirstResponder()
        return true
    }
    func textFieldDidEndEditing(_ textField: UITextField) {
        if let text = textField.text,
           let rate = Float(text) {
            if rate > 2.0 {
                textField.text = nil
                //debugPrint("The maximum rate is 2.0")
                return;
            }
            self.rate = rate
        } else {
            //debugPrint("The rate should be a number")
        }
    }
    
    // MARK: Functions
    
    func totalTime() -> TimeInterval {
        if let player, let item = player.currentItem {
            let sec = CMTimeGetSeconds(item.duration)
            if !sec.isNaN {
                return TimeInterval(sec)
            }
        }
        return 0
    }

    func currentTime() -> TimeInterval {
        if let player {
            let sec = CMTimeGetSeconds(player.currentTime())
            if !sec.isNaN {
                return TimeInterval(sec)
            }
        }
        return 0
    }

    func secondsConvertedToMinuteString(seconds: Int) -> String {
        let minite = seconds / 60
        let redundant = seconds % 60
        return String(format: "%02d:%02d", minite, redundant)
    }
}
