//
//  ExampleForPatternVC.m
//  LovenseDemo
//
//  Created by 陈自豪 on 2023/5/26.
//  Copyright © 2023 June. All rights reserved.
//

#import "ExampleForPatternVC.h"
#import <AVFoundation/AVFoundation.h>
#import <Lovense/Lovense.h>
@interface ExampleForPatternVC () <UITextFieldDelegate>
// player
@property(nonatomic, strong) AVPlayer * player;
// playItem
@property(nonatomic, strong) AVPlayerItem * playerItem;
// playerLayer
@property(nonatomic, strong) AVPlayerLayer * playerLayer;
// current Time
@property(nonatomic, weak) UILabel * currentTimeLabel;
// total Time
@property(nonatomic, weak) UILabel * totalTimeLabel;
// Sliser
@property(nonatomic, weak) UISlider * slider;
// Rate
@property(nonatomic, assign) CGFloat rate;

@property(nonatomic, assign) bool isDragSlider;


@property(nonatomic, weak) UITextField * textField;


@end

@implementation ExampleForPatternVC {
    id _timeObserver;
}
-  (void)dealloc {
    [self.player pause];
    self.player = nil;
    [self.player removeTimeObserver:_timeObserver];
    _timeObserver = nil;
    NSLog(@"ExampleForPatternVC delloc");
}
- (void)viewDidLoad {
    [super viewDidLoad];
    [self setupUI];
    
    self.rate = 1.0;
    // Initialize your toys and platform information
    NSMutableArray * toyIds = [NSMutableArray array];
    
    [Lovense.shared setupPatternPlayerWithToyIds: toyIds pf:@"Your platform"];
    
    __weak typeof (self) weakSelf = self;
    CMTime interval = CMTimeMakeWithSeconds(1 , NSEC_PER_SEC);
    _timeObserver = [self.player addPeriodicTimeObserverForInterval: interval queue:dispatch_get_main_queue() usingBlock:^(CMTime time) {
            NSArray *loadedRanges = weakSelf.playerItem.seekableTimeRanges;
            if (loadedRanges.count > 0) {
                if (weakSelf.isDragSlider) {
                    weakSelf.isDragSlider = NO;
                } else {
                    NSTimeInterval currentTime = [weakSelf currentTime];
                    NSTimeInterval totalTime = [weakSelf totalTime];
                    if (currentTime == totalTime) {
                        weakSelf.slider.value = 0;
                        [weakSelf.player seekToTime:kCMTimeZero];
                        [weakSelf.player pause];
                        weakSelf.currentTimeLabel.text = @"0:00";
                        [Lovense.shared stopPattern];
                        
                    } else {
                        weakSelf.slider.value = currentTime / totalTime;
                        weakSelf.currentTimeLabel.text = [weakSelf secondsConvertedToMinuteStringWithSecond:currentTime];
                        weakSelf.totalTimeLabel.text = [weakSelf secondsConvertedToMinuteStringWithSecond:totalTime];
                    }
                    
                }

            }
    }];
    
}

- (void) setupUI {
    self.view.backgroundColor = [UIColor whiteColor];
    
    self.title = @"Pattern Play";
    self.playerItem = [AVPlayerItem playerItemWithURL:[NSURL URLWithString:@"Your media URL"]];
    self.player = [AVPlayer playerWithPlayerItem: self.playerItem];
    self.playerLayer = [AVPlayerLayer playerLayerWithPlayer:self.player];
    self.playerLayer.frame = CGRectMake(0, 80, self.view.bounds.size.width, 200);
    self.playerLayer.backgroundColor = [[UIColor grayColor] CGColor];
    [self.view.layer addSublayer: self.playerLayer];
    

    
    UIButton * playButton = [UIButton new];
    [playButton setTitle:@"Play" forState: UIControlStateNormal];
    [playButton setTitleColor:[UIColor blueColor] forState: UIControlStateNormal];
    [playButton setTitleColor:[UIColor redColor] forState: UIControlStateHighlighted];
    playButton.layer.borderColor = [[UIColor redColor] CGColor];
    playButton.layer.cornerRadius = 3;
    playButton.layer.borderWidth = 1;
    playButton.titleLabel.font = [UIFont systemFontOfSize:15];
    [self.view addSubview:playButton];
    [playButton addTarget:self action:@selector(clickPlayButton) forControlEvents: UIControlEventTouchUpInside];
    playButton.frame = CGRectMake(50, 300, 80, 40);
    
    UIButton * pauseButton = [UIButton new];
    [pauseButton setTitle:@"Pause" forState: UIControlStateNormal];
    [pauseButton setTitleColor:[UIColor blueColor] forState: UIControlStateNormal];
    [pauseButton setTitleColor:[UIColor redColor] forState: UIControlStateHighlighted];
    pauseButton.titleLabel.font = [UIFont systemFontOfSize:15];
    pauseButton.layer.borderColor = [[UIColor redColor] CGColor];
    pauseButton.layer.cornerRadius = 3;
    pauseButton.layer.borderWidth = 1;
    [self.view addSubview:pauseButton];
    [pauseButton addTarget:self action:@selector(clickPauseButton) forControlEvents: UIControlEventTouchUpInside];
    pauseButton.frame = CGRectMake(150, 300, 80, 40);
    
    UIButton * stopButton = [UIButton new];
    [stopButton setTitle:@"Stop" forState: UIControlStateNormal];
    [stopButton setTitleColor:[UIColor blueColor] forState: UIControlStateNormal];
    [stopButton setTitleColor:[UIColor redColor] forState: UIControlStateHighlighted];
    stopButton.titleLabel.font = [UIFont systemFontOfSize:15];
    stopButton.layer.borderColor = [[UIColor redColor] CGColor];
    stopButton.layer.cornerRadius = 3;
    stopButton.layer.borderWidth = 1;
    [self.view addSubview:stopButton];
    [stopButton addTarget:self action:@selector(clickStopButton) forControlEvents: UIControlEventTouchUpInside];
    stopButton.frame = CGRectMake(250, 300, 80, 40);
    
    UILabel * label = [[UILabel alloc]init];
    label.text = @"Set Rate(0 ~ 2.0):";
    label.textColor = [UIColor blackColor];
    [self.view addSubview:label];
    label.frame =  CGRectMake(20, 370, 150, 40);
    
    
    UITextField * textField = [[UITextField alloc] init];
    textField.keyboardType = UIKeyboardTypeNumbersAndPunctuation;
    textField.returnKeyType = UIReturnKeyDone;
    textField.placeholder = @"Default is 1.0";
    textField.delegate = self;
    textField.borderStyle = UITextBorderStyleRoundedRect;
    textField.textColor = [UIColor blueColor];
    [self.view addSubview:textField];
    textField.frame =  CGRectMake(20, 415, 220, 40);
    self.textField = textField;
    
    UILabel * label1 = [[UILabel alloc]init];
    label1.text = @"Drag here to adjust the playback";
    label1.textColor = [UIColor blackColor];
    [self.view addSubview:label1];
    label1.frame =  CGRectMake(20, 470, 300, 40);
    
    
    UILabel * currentTimeLabel = [[UILabel alloc]init];
    currentTimeLabel.textAlignment = NSTextAlignmentRight;
    currentTimeLabel.text = @"--:--";
    currentTimeLabel.textColor = [UIColor blackColor];
    [self.view addSubview:currentTimeLabel];
    currentTimeLabel.frame =  CGRectMake(10, 517, 47, 20);
    self.currentTimeLabel = currentTimeLabel;
    
    UISlider *slider = [[UISlider alloc] init];
    slider.minimumValue = 0;
    slider.maximumValue = 1;
    slider.value = 0;
    [slider addTarget:self action:@selector(sliderValueChanged:) forControlEvents:UIControlEventTouchUpInside];
    [self.view addSubview:slider];
    slider.frame = CGRectMake(70, 510, 230, 40);
    self.slider = slider;
    
    UILabel * totalTimeLabel = [[UILabel alloc]init];
    totalTimeLabel.textAlignment = NSTextAlignmentLeft;
    totalTimeLabel.text = @"--:--";
    totalTimeLabel.textColor = [UIColor blackColor];
    [self.view addSubview:totalTimeLabel];
    totalTimeLabel.frame =  CGRectMake(310, 517, 50, 20);
    self.totalTimeLabel = totalTimeLabel;
    
    
}

- (void)sliderValueChanged: (UISlider*) slider {
    CGFloat progress = slider.value / slider.maximumValue;
    NSTimeInterval currentTime = [self totalTime] * progress;
    [self.player seekToTime:CMTimeMake(currentTime, 1)];
    // sync current time
    self.isDragSlider = YES;
    [Lovense.shared syncCurrentTime: currentTime * 1000];
}


- (NSTimeInterval) totalTime {
    NSTimeInterval sec = CMTimeGetSeconds(self.player.currentItem.duration);
    if (isnan(sec)) {
        return 0;
    }
    return sec;
}

- (NSTimeInterval) currentTime {
    NSTimeInterval sec = CMTimeGetSeconds(self.playerItem.currentTime);
    if (isnan(sec) || sec < 0) {
        return 0;
    }
    return sec;
}

#pragma mark - UITextFieldDelegate
- (BOOL)textFieldShouldReturn:(UITextField *)textField {
    [textField resignFirstResponder];
    return true;
}
- (void)textFieldDidEndEditing:(UITextField *)textField {
    if (textField == self.textField) {
        CGFloat rate = [textField.text floatValue];
        if (rate > 2.0 ){
            textField.text = nil;
            NSLog(@"The maximum Rate is 2.0");
            return;
        }
        self.rate = rate;
    }
    
}


#pragma  mark - Action
- (void) clickPlayButton {
    [self.player play];
    self.player.rate = self.rate;
    [Lovense.shared setRate: self.rate];
    /// prepare pattern file
    __weak typeof (self) weakSelf = self;
    [Lovense.shared prepareForPatternWithMediaID: @"Your mediaId" completion:^(PatternPrepareState state) {
        __strong __typeof(weakSelf)strongSelf = weakSelf;
        if (state == PatternPrepareStateDownloading) {
            // ...
        } else if (state == PatternPrepareStateDownloadCompleted) {
            // play pattern
            [Lovense.shared playPatternWithCurrentTime:[strongSelf currentTime] * 1000 andTotalTime:[strongSelf totalTime] * 1000];
        } else if (state == PatternPrepareStateDownloadFail) {
            // do something
            UIAlertController * alert = [UIAlertController alertControllerWithTitle:@"notice" message:@"pattern file download fail" preferredStyle:UIAlertControllerStyleAlert];
            UIAlertAction * action1 = [UIAlertAction actionWithTitle:@"OK" style:UIAlertActionStyleCancel handler:nil];
            [alert addAction:action1];
            [strongSelf presentViewController:alert animated:true completion:nil];
        }
    }];
}

#pragma mark - Other
- (void) clickPauseButton {
    [self.player pause];
    [Lovense.shared  pausePattern];
}

- (void) clickStopButton {
    [self.player pause];
    [self.player seekToTime:kCMTimeZero];
    [Lovense.shared  stopPattern];
}

- (NSString * )secondsConvertedToMinuteStringWithSecond: (int) seconds {
    int minite = seconds / 60 ;
    int redundant = seconds % 60;
  
    return [NSString stringWithFormat:@"%d:%d",minite,redundant];
}
@end
